package MusicLandscape.entities;

/**
 * This class represents a release of a specific artist, like an album or a music video.
 *
 * @author Jonas Altrock (ew20b126@technikum-wien.at)
 * @version 1
 * @since ExerciseSheet04
 */
public abstract class Release {
    /**
     * The artist that released this release.<br>
     * A null artist represents an unknown artist. By default, it is the default artist.
     */
    private Artist artist = new Artist();

    /**
     * The year in which this release was released.<br>
     * The year cannot be negative, value 0 (zero) representing unknown year. By default, the year is 1900.
     */
    private int year = 1900;

    /**
     * The title of this release.<br>
     * A null title represents an unknown title. The default title is unknown.
     */
    private String title = null;

    /**
     * Creates a default release.<br>
     * A default release has default values for all fields (see there).
     */
    public Release() {

    }

    /**
     * Creates a copy of a release.<br>
     * This release is a deep copy of an existing release - the original.
     *
     * @param r the release to be copied
     */
    public Release(Release r) {
        artist = new Artist(r.artist);
        year = r.year;
        title = r.title;
    }

    /**
     * Create a release with a specific title of a specific artist in a specific year.
     *
     * @param title  the title of the new release
     * @param artist the artist of the new release
     * @param year   the year of the new release
     */
    public Release(String title, Artist artist, int year) {
        this.title = title;
        this.artist = artist;
        this.year = year;
    }

    /**
     * Get the title of this release.<br>
     * If the title is unknown a null String is returned.
     *
     * @return the title
     */
    public String getTitle() {
        return title;
    }


    /**
     * Set the title of this release.<br>
     * This method accepts null Strings.
     *
     * @param title the title to set
     */
    public void setTitle(String title) {
        this.title = title;
    }

    /**
     * Get the artist of this release.
     *
     * @return the artist of this release.
     */
    public Artist getArtist() {
        return artist;
    }

    /**
     * Set the artist of this release.<br>
     * This method accepts null arguments.
     *
     * @param artist the artist to set.
     */
    public void setArtist(Artist artist) {
        this.artist = artist;
    }

    /**
     * Get the year in which this release was released.
     *
     * @return the year of this release.
     */
    public int getYear() {
        return year;
    }

    /**
     * Set the year of this release.<br>
     * If the argument is outside the allowed range it is ignored.
     *
     * @param year the year to set.
     */
    public void setYear(int year) {
        if (year != 0 && year < 1900) {
            return;
        }
        this.year = year;
    }

    /**
     * Get the total time of this release.<br>
     * The implementation of this method is left to concrete subclasses.
     *
     * @return the duration of this release in seconds.
     */
    public abstract int totalTime();

    /**
     * Get a textual representation of this release.
     * <p>
     * The string representation of a release is "title-artist-year-totaltime" (without quotes and all names
     * substituted with their respective values) unknown field are represented by the string "unknown" (without quotes).
     *
     * @return the string representation
     */
    @Override
    public String toString() {
        String title = this.title == null || this.title.isBlank() ? "unknown" : this.title;
        String artist = this.artist == null || this.artist.getName().isBlank() ? "unknown" : this.artist.getName();
        String year = this.year == 0 ? "unknown" : this.year + "";

        return title + "-" + artist + "-" + year + "-" + totalTime();
    }
}
